/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jena.http;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpHeaders;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.zip.GZIPInputStream;
import java.util.zip.InflaterInputStream;
import org.apache.commons.compress.compressors.bzip2.BZip2CompressorInputStream;
import org.apache.jena.atlas.RuntimeIOException;
import org.apache.jena.atlas.io.IO;
import org.apache.jena.atlas.lib.IRILib;
import org.apache.jena.atlas.lib.InternalErrorException;
import org.apache.jena.atlas.logging.FmtLog;
import org.apache.jena.atlas.web.HttpException;
import org.apache.jena.atlas.web.TypedInputStream;
import org.apache.jena.http.AsyncHttpRDF;
import org.apache.jena.http.HttpEnv;
import org.apache.jena.http.Push;
import org.apache.jena.http.auth.AuthEnv;
import org.apache.jena.http.auth.AuthLib;
import org.apache.jena.http.sys.HttpRequestModifier;
import org.apache.jena.http.sys.RegistryRequestModifier;
import org.apache.jena.query.ARQ;
import org.apache.jena.sparql.exec.http.Params;
import org.apache.jena.sparql.util.Context;
import org.apache.jena.web.HttpSC;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HttpLib {
    private static Logger LOG = LoggerFactory.getLogger((String)(HttpLib.class.getPackageName() + ".HTTP"));
    private static HttpResponse.BodyHandler<InputStream> bodyHandlerInputStream = HttpLib.buildDftBodyHandlerInputStream();
    private static Function<HttpResponse<InputStream>, String> bodyInputStreamToString = r -> {
        try {
            InputStream in = (InputStream)r.body();
            String msg = IO.readWholeFileAsUTF8((InputStream)in);
            return msg;
        }
        catch (Throwable ex) {
            throw new HttpException(ex);
        }
    };
    public static String BEARER = "Bearer";
    public static String BEARER_PREFIX = BEARER + " ";
    private static final int SKIP_BUFFER_SIZE = 16384;
    private static final byte[] SKIP_BYTE_BUFFER_WO = new byte[16384];
    public static String FusekiRequestIdHeader = "Fuseki-Request-Id";

    private HttpLib() {
    }

    public static HttpResponse.BodyHandler<Void> noBody() {
        return HttpResponse.BodyHandlers.discarding();
    }

    public static HttpRequest.BodyPublisher stringBody(String str2) {
        return HttpRequest.BodyPublishers.ofString(str2);
    }

    private static HttpResponse.BodyHandler<InputStream> buildDftBodyHandlerInputStream() {
        return responseInfo -> HttpResponse.BodySubscribers.ofInputStream();
    }

    public static String basicAuth(String username, String password) {
        return "Basic " + Base64.getEncoder().encodeToString((username + ":" + password).getBytes(StandardCharsets.UTF_8));
    }

    public static String bearerAuthHeader(String tokenBase64) {
        Objects.requireNonNull(tokenBase64);
        if (tokenBase64.indexOf(32) >= 0) {
            throw new IllegalArgumentException("Base64 token contains a space");
        }
        return BEARER_PREFIX + tokenBase64;
    }

    public static String bearerAuthTokenFromHeader(String authHeaderString) {
        Objects.requireNonNull(authHeaderString);
        if (!authHeaderString.startsWith(BEARER_PREFIX)) {
            throw new IllegalArgumentException("Auth headerString does not start 'Bearer ...'");
        }
        return authHeaderString.substring("Bearer ".length()).trim();
    }

    public static InputStream getInputStream(HttpResponse<InputStream> httpResponse) {
        String encoding = httpResponse.headers().firstValue("Content-Encoding").orElse("");
        InputStream responseInput = httpResponse.body();
        try {
            switch (encoding) {
                case "": 
                case "identity": {
                    return responseInput;
                }
                case "gzip": {
                    return new GZIPInputStream(responseInput, 2048);
                }
                case "bzip2": {
                    return new BZip2CompressorInputStream(responseInput, true);
                }
                case "inflate": {
                    return new InflaterInputStream(responseInput);
                }
            }
            throw new UnsupportedOperationException("Not supported: Content-Encoding: " + encoding);
        }
        catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    public static void handleHttpStatusCode(HttpResponse<InputStream> response) {
        int httpStatusCode = response.statusCode();
        if (!HttpLib.inRange(httpStatusCode, 100, 599)) {
            throw new HttpException("Status code out of range: " + httpStatusCode);
        }
        if (HttpLib.inRange(httpStatusCode, 100, 199)) {
            return;
        }
        if (HttpLib.inRange(httpStatusCode, 200, 299)) {
            return;
        }
        if (HttpLib.inRange(httpStatusCode, 300, 399)) {
            throw HttpLib.exception(response, httpStatusCode);
        }
        if (HttpLib.inRange(httpStatusCode, 400, 499)) {
            throw HttpLib.exception(response, httpStatusCode);
        }
        if (HttpLib.inRange(httpStatusCode, 500, 599)) {
            throw HttpLib.exception(response, httpStatusCode);
        }
        throw new InternalErrorException("Unknown status code: " + httpStatusCode);
    }

    public static InputStream handleResponseInputStream(HttpResponse<InputStream> httpResponse) {
        HttpLib.handleHttpStatusCode(httpResponse);
        return HttpLib.getInputStream(httpResponse);
    }

    public static TypedInputStream handleResponseTypedInputStream(HttpResponse<InputStream> httpResponse) {
        InputStream input = HttpLib.handleResponseInputStream(httpResponse);
        String ct = HttpLib.responseHeader(httpResponse, "Content-Type");
        return new TypedInputStream(input, ct);
    }

    public static void handleResponseNoBody(HttpResponse<InputStream> response) {
        HttpLib.handleHttpStatusCode(response);
        HttpLib.finishResponse(response);
    }

    public static String handleResponseRtnString(HttpResponse<InputStream> response) {
        return HttpLib.handleResponseRtnString(response, null);
    }

    public static String handleResponseRtnString(HttpResponse<InputStream> response, Consumer<InputStream> callback) {
        InputStream input = HttpLib.handleResponseInputStream(response);
        if (callback != null) {
            callback.accept(input);
        }
        try {
            String string = IO.readWholeFileAsUTF8((InputStream)input);
            return string;
        }
        catch (RuntimeIOException e2) {
            throw new HttpException(e2);
        }
        finally {
            HttpLib.finishInputStream(input);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static HttpException exception(HttpResponse<InputStream> response, int httpStatusCode) {
        InputStream in = response.body();
        if (in == null) {
            return new HttpException(httpStatusCode, HttpSC.getMessage(httpStatusCode));
        }
        try {
            String msg;
            try {
                msg = IO.readWholeFileAsUTF8((InputStream)in);
                if (msg.isBlank()) {
                    msg = null;
                }
            }
            catch (RuntimeIOException e2) {
                msg = null;
            }
            HttpException httpException = new HttpException(httpStatusCode, HttpSC.getMessage(httpStatusCode), msg);
            return httpException;
        }
        finally {
            IO.close((Closeable)in);
        }
    }

    private static long getContentLength(HttpResponse<InputStream> response) {
        Optional<String> x = response.headers().firstValue("Content-Length");
        if (x.isEmpty()) {
            return -1L;
        }
        try {
            return Long.parseLong(x.orElseThrow());
        }
        catch (NumberFormatException ex) {
            return -1L;
        }
    }

    private static boolean inRange(int x, int min2, int max2) {
        return min2 <= x && x <= max2;
    }

    public static void finishResponse(HttpResponse<InputStream> response) {
        InputStream input = response.body();
        if (input == null) {
            return;
        }
        HttpLib.finishInputStream(input);
    }

    public static void finishInputStream(InputStream input) {
        HttpLib.consumeAndClose(input);
    }

    private static void consumeAndClose(InputStream input) {
        long bytesRead = 0L;
        try {
            long n;
            while ((n = (long)input.read(SKIP_BYTE_BUFFER_WO, 0, 16384)) >= 0L) {
                bytesRead += n;
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
        try {
            input.close();
        }
        catch (Throwable throwable) {
            // empty catch block
        }
    }

    public static URI toRequestURI(String uriStr) {
        try {
            URI uri = new URI(uriStr);
            if (!uri.isAbsolute()) {
                throw new HttpException("Not an absolute URL: <" + uriStr + ">");
            }
            return uri;
        }
        catch (URISyntaxException ex) {
            int idx = ex.getIndex();
            String msg = idx < 0 ? String.format("Bad URL: %s", uriStr) : String.format("Bad URL: %s starting at character %d", uriStr, idx);
            throw new HttpException(msg, ex);
        }
    }

    public static boolean isEndpoint(URI uri) {
        return uri.isAbsolute() && uri.getHost() != null && uri.getRawPath() != null && uri.getRawQuery() == null && uri.getRawFragment() == null;
    }

    public static String endpoint(String uriStr) {
        int idx2;
        int idx1 = uriStr.indexOf(63);
        int n = idx2 = idx1 < 0 ? uriStr.indexOf(35) : -1;
        if (idx1 < 0 && idx2 < 0) {
            return uriStr;
        }
        int idx = -1;
        idx = idx1 < 0 && idx2 > 0 ? idx2 : (idx1 > 0 && idx2 < 0 ? idx1 : Math.min(idx1, idx2));
        return uriStr.substring(0, idx);
    }

    public static URI endpointURI(URI uri) {
        if (uri.getRawQuery() == null && uri.getRawFragment() == null) {
            return uri;
        }
        try {
            return new URI(uri.getScheme(), uri.getRawAuthority(), uri.getRawPath(), null, null);
        }
        catch (URISyntaxException x) {
            throw new IllegalArgumentException(x.getMessage(), x);
        }
    }

    public static String requestTargetServer(URI uri) {
        String path = uri.getRawPath();
        if (path == null || path.isEmpty()) {
            path = "/";
        }
        return path;
    }

    public static HttpRequest newGetRequest(String url, Consumer<HttpRequest.Builder> modifier) {
        HttpRequest.Builder builder = HttpLib.requestBuilderFor(url).uri(HttpLib.toRequestURI(url)).GET();
        if (modifier != null) {
            modifier.accept(builder);
        }
        return builder.build();
    }

    public static <X> X dft(X value, X dftValue) {
        return value != null ? value : dftValue;
    }

    public static <X> List<X> copyArray(List<X> array) {
        if (array == null) {
            return null;
        }
        return new ArrayList<X>(array);
    }

    public static String urlEncodeQueryString(String str2) {
        String x1 = IRILib.encodeUriQueryFrag((String)str2);
        String x2 = IRILib.encodeNonASCII((String)x1);
        return x2;
    }

    public static String requestURL(String url, String queryString) {
        if (queryString == null || queryString.isEmpty()) {
            return url;
        }
        String sep = url.contains("?") ? "&" : "?";
        String requestURL = url + sep + queryString;
        return requestURL;
    }

    public static HttpRequest.Builder requestBuilderFor(String serviceEndpoint) {
        HttpRequest.Builder requestBuilder = HttpRequest.newBuilder();
        return AuthEnv.get().addAuth(requestBuilder, serviceEndpoint);
    }

    public static HttpRequest.Builder requestBuilder(String url, Map<String, String> httpHeaders, long readTimeout, TimeUnit readTimeoutUnit) {
        HttpRequest.Builder builder = HttpLib.requestBuilderFor(url);
        HttpLib.headers(builder, httpHeaders);
        builder.uri(HttpLib.toRequestURI(url));
        if (readTimeout >= 0L) {
            builder.timeout(Duration.ofMillis(readTimeoutUnit.toMillis(readTimeout)));
        }
        return builder;
    }

    public static HttpRequest.Builder createBuilder(HttpRequest request) {
        HttpRequest.Builder builder = HttpRequest.newBuilder().expectContinue(request.expectContinue()).uri(request.uri());
        builder.method(request.method(), request.bodyPublisher().orElse(HttpRequest.BodyPublishers.noBody()));
        request.timeout().ifPresent(builder::timeout);
        request.version().ifPresent(builder::version);
        request.headers().map().forEach((name, values) -> values.forEach(value -> builder.header((String)name, (String)value)));
        return builder;
    }

    static HttpRequest.Builder headers(HttpRequest.Builder builder, Map<String, String> httpHeaders) {
        if (httpHeaders != null) {
            httpHeaders.forEach(builder::header);
        }
        return builder;
    }

    public static HttpRequest.Builder acceptHeader(HttpRequest.Builder builder, String acceptHeader) {
        if (acceptHeader != null) {
            builder.header("Accept", acceptHeader);
        }
        return builder;
    }

    public static HttpRequest.Builder contentTypeHeader(HttpRequest.Builder builder, String contentType) {
        if (contentType != null) {
            builder.header("Content-Type", contentType);
        }
        return builder;
    }

    public static HttpResponse<InputStream> execute(HttpClient httpClient, HttpRequest httpRequest) {
        return AsyncHttpRDF.getOrElseThrow(HttpLib.executeAsync(httpClient, httpRequest, HttpResponse.BodyHandlers.ofInputStream()), httpRequest);
    }

    public static CompletableFuture<HttpResponse<InputStream>> executeAsync(HttpClient httpClient, HttpRequest httpRequest) {
        return HttpLib.executeAsync(httpClient, httpRequest, HttpResponse.BodyHandlers.ofInputStream());
    }

    static <X> CompletableFuture<HttpResponse<X>> executeAsync(HttpClient httpClient, HttpRequest httpRequest, HttpResponse.BodyHandler<X> bodyHandler) {
        String[] userpasswd;
        URI uri = httpRequest.uri();
        URI key = null;
        AuthEnv authEnv = AuthEnv.get();
        if (uri.getUserInfo() != null && (userpasswd = uri.getUserInfo().split(":")).length == 2) {
            key = HttpLib.endpointURI(uri);
            authEnv.registerUsernamePassword(key, userpasswd[0], userpasswd[1]);
        }
        URI finalKey = key;
        return AuthLib.authExecuteAsync(httpClient, httpRequest, bodyHandler).whenComplete((httpResponse, throwable) -> {
            if (finalKey != null) {
                authEnv.unregisterUsernamePassword(finalKey);
            }
        });
    }

    public static <T> HttpResponse<T> executeJDK(HttpClient httpClient, HttpRequest httpRequest, HttpResponse.BodyHandler<T> bodyHandler) {
        return AsyncHttpRDF.getOrElseThrow(HttpLib.executeJDKAsync(httpClient, httpRequest, bodyHandler), httpRequest);
    }

    public static CompletableFuture<HttpResponse<InputStream>> executeJDKAsync(HttpClient httpClient, HttpRequest httpRequest) {
        return HttpLib.executeAsync(httpClient, httpRequest, HttpResponse.BodyHandlers.ofInputStream());
    }

    public static <T> CompletableFuture<HttpResponse<T>> executeJDKAsync(HttpClient httpClient, HttpRequest httpRequest, HttpResponse.BodyHandler<T> bodyHandler) {
        HttpLib.logRequest(httpRequest);
        CompletionStage future = httpClient.sendAsync(httpRequest, bodyHandler).thenApply(httpResponse -> {
            HttpLib.logResponse(httpResponse);
            return httpResponse;
        });
        return future;
    }

    static CompletableFuture<HttpResponse<InputStream>> asyncExecute(HttpClient httpClient, HttpRequest httpRequest) {
        HttpLib.logAsyncRequest(httpRequest);
        return httpClient.sendAsync(httpRequest, HttpResponse.BodyHandlers.ofInputStream());
    }

    public static void httpPushData(HttpClient httpClient, Push style, String url, Consumer<HttpRequest.Builder> modifier, HttpRequest.BodyPublisher body) {
        HttpResponse<InputStream> response = HttpLib.httpPushWithResponse(httpClient, style, url, modifier, body);
        HttpLib.handleResponseNoBody(response);
    }

    public static HttpResponse<InputStream> httpPushWithResponse(HttpClient httpClient, Push style, String url, Consumer<HttpRequest.Builder> modifier, HttpRequest.BodyPublisher body) {
        URI uri = HttpLib.toRequestURI(url);
        HttpRequest.Builder builder = HttpLib.requestBuilderFor(url);
        builder.uri(uri);
        builder.method(style.method(), body);
        if (modifier != null) {
            modifier.accept(builder);
        }
        HttpResponse<InputStream> response = HttpLib.execute(httpClient, builder.build());
        return response;
    }

    private static void logRequest(HttpRequest httpRequest) {
        if (LOG.isDebugEnabled()) {
            FmtLog.debug((Logger)LOG, (String)"> %s %s", (Object[])new Object[]{httpRequest.method(), httpRequest.uri()});
            HttpLib.logHeaders(LOG, httpRequest.headers());
        }
    }

    private static void logAsyncRequest(HttpRequest httpRequest) {
    }

    private static void logResponse(HttpResponse<?> httpResponse) {
        if (LOG.isDebugEnabled()) {
            FmtLog.debug((Logger)LOG, (String)"< %d %s %s", (Object[])new Object[]{httpResponse.statusCode(), httpResponse.request().method(), httpResponse.uri()});
            HttpLib.logHeaders(LOG, httpResponse.headers());
        }
    }

    private static void logHeaders(Logger log2, HttpHeaders headers) {
        headers.map().forEach((header, values) -> values.forEach(value -> FmtLog.debug((Logger)log2, (String)"  %-15s %s", (Object[])new Object[]{header, value})));
    }

    public static void modifyByService(String serviceURI, Context context2, Params params, Map<String, String> httpHeaders) {
        HttpRequestModifier mods;
        HttpRequestModifier modifier = (HttpRequestModifier)context2.get(ARQ.httpRequestModifer);
        if (modifier != null) {
            modifier.modify(params, httpHeaders);
            return;
        }
        RegistryRequestModifier modifierRegistry = (RegistryRequestModifier)context2.get(ARQ.httpRegistryRequestModifer);
        if (modifierRegistry == null) {
            modifierRegistry = RegistryRequestModifier.get();
        }
        if (modifierRegistry != null && (mods = (HttpRequestModifier)modifierRegistry.find(serviceURI)) != null) {
            mods.modify(params, httpHeaders);
        }
    }

    public static Consumer<HttpRequest.Builder> setHeaders(Map<String, String> headers) {
        if (headers == null) {
            return x -> {};
        }
        return x -> headers.forEach(x::header);
    }

    static Consumer<HttpRequest.Builder> setAcceptHeader(String acceptHeader) {
        if (acceptHeader == null) {
            return x -> {};
        }
        return HttpLib.header("Accept", acceptHeader);
    }

    static Consumer<HttpRequest.Builder> setContentTypeHeader(String contentType) {
        if (contentType == null) {
            return x -> {};
        }
        return HttpLib.header("Content-Type", contentType);
    }

    static Consumer<HttpRequest.Builder> header(String headerName, String headerValue) {
        return x -> x.header(headerName, headerValue);
    }

    public static String responseHeader(HttpResponse<?> response, String headerName) {
        Objects.requireNonNull(response);
        Objects.requireNonNull(headerName);
        return response.headers().firstValue(headerName).orElse(null);
    }

    public static boolean isFuseki(String datasetURL) {
        HttpRequest.Builder builder = HttpRequest.newBuilder().uri(HttpLib.toRequestURI(datasetURL)).method("HEAD", HttpRequest.BodyPublishers.noBody());
        HttpRequest request = builder.build();
        HttpClient httpClient = HttpEnv.getDftHttpClient();
        HttpResponse<InputStream> response = HttpLib.execute(httpClient, request);
        HttpLib.handleResponseNoBody(response);
        Optional<String> value1 = response.headers().firstValue(FusekiRequestIdHeader);
        if (value1.isPresent()) {
            return true;
        }
        Optional<String> value2 = response.headers().firstValue("Server");
        if (value2.isEmpty()) {
            return false;
        }
        String headerValue = value2.orElseThrow();
        boolean isFuseki = headerValue.startsWith("Apache Jena Fuseki") || headerValue.toLowerCase().contains("fuseki");
        return isFuseki;
    }
}

